const CACHE_VERSION = 'v2.1.0';
const CACHE_NAME = `plasiyer-${CACHE_VERSION}`;
const STATIC_CACHE = `plasiyer-static-${CACHE_VERSION}`;
const DYNAMIC_CACHE = `plasiyer-dynamic-${CACHE_VERSION}`;
const OFFLINE_QUEUE = 'plasiyer-offline-queue';

// Static assets to cache immediately
const STATIC_ASSETS = [
    '/',
    '/index.html',
    '/manifest.json',
    '/icons/icon-192x192.png',
    '/icons/icon-512x512.png',
];

// URLs to always fetch from network (never cache)
const NETWORK_ONLY = [
    '/api/',
];

// URLs that can be cached with network-first strategy
const NETWORK_FIRST = [
    '/api/dashboard',
    '/api/reports',
];

// Install event - cache static assets
self.addEventListener('install', (event) => {
    console.log('[SW] Installing Service Worker...', CACHE_VERSION);
    event.waitUntil(
        caches.open(STATIC_CACHE)
            .then((cache) => {
                console.log('[SW] Caching static assets');
                return cache.addAll(STATIC_ASSETS);
            })
            .then(() => {
                console.log('[SW] Static assets cached');
                return self.skipWaiting();
            })
            .catch((err) => {
                console.error('[SW] Failed to cache static assets:', err);
            })
    );
});

// Activate event - clean old caches
self.addEventListener('activate', (event) => {
    console.log('[SW] Activating Service Worker...', CACHE_VERSION);
    event.waitUntil(
        caches.keys()
            .then((cacheNames) => {
                return Promise.all(
                    cacheNames
                        .filter((name) => {
                            // Delete old version caches
                            return name.startsWith('plasiyer-') &&
                                   name !== STATIC_CACHE &&
                                   name !== DYNAMIC_CACHE &&
                                   name !== OFFLINE_QUEUE;
                        })
                        .map((name) => {
                            console.log('[SW] Deleting old cache:', name);
                            return caches.delete(name);
                        })
                );
            })
            .then(() => {
                console.log('[SW] Service Worker activated');
                return self.clients.claim();
            })
    );
});

// Helper: Check if URL matches any pattern
function matchesPattern(url, patterns) {
    return patterns.some(pattern => url.includes(pattern));
}

// Helper: Clone response for caching
function cloneResponse(response) {
    if (!response || response.status !== 200 || response.type !== 'basic') {
        return null;
    }
    return response.clone();
}

// Fetch event - serve from cache, fallback to network
self.addEventListener('fetch', (event) => {
    const { request } = event;
    const url = new URL(request.url);

    // Skip non-http(s) requests
    if (!url.protocol.startsWith('http')) {
        return;
    }

    // Skip non-GET requests
    if (request.method !== 'GET') {
        return;
    }

    // Skip chrome-extension and other non-same-origin requests
    if (url.origin !== self.location.origin) {
        return;
    }

    // API requests - Network only with offline fallback
    if (url.pathname.startsWith('/api')) {
        event.respondWith(
            fetch(request)
                .then((response) => {
                    // Optionally cache successful API responses for offline fallback
                    if (response.ok && matchesPattern(url.pathname, NETWORK_FIRST)) {
                        const responseClone = response.clone();
                        caches.open(DYNAMIC_CACHE)
                            .then((cache) => cache.put(request, responseClone));
                    }
                    return response;
                })
                .catch(async () => {
                    // Try to get from cache for network-first APIs
                    if (matchesPattern(url.pathname, NETWORK_FIRST)) {
                        const cached = await caches.match(request);
                        if (cached) {
                            return cached;
                        }
                    }
                    // Return offline error response
                    return new Response(
                        JSON.stringify({
                            error: 'Çevrimdışı',
                            message: 'İnternet bağlantısı bulunamadı. Lütfen bağlantınızı kontrol edin.',
                            offline: true
                        }),
                        {
                            status: 503,
                            headers: {
                                'Content-Type': 'application/json',
                                'X-Offline': 'true'
                            }
                        }
                    );
                })
        );
        return;
    }

    // Static assets - Cache First, then Network
    event.respondWith(
        caches.match(request)
            .then((cachedResponse) => {
                if (cachedResponse) {
                    // Return cached version, update cache in background (stale-while-revalidate)
                    event.waitUntil(
                        fetch(request)
                            .then((networkResponse) => {
                                if (networkResponse && networkResponse.ok) {
                                    caches.open(DYNAMIC_CACHE)
                                        .then((cache) => cache.put(request, networkResponse));
                                }
                            })
                            .catch(() => { /* Ignore network errors for background update */ })
                    );
                    return cachedResponse;
                }

                // Not in cache - fetch from network
                return fetch(request)
                    .then((networkResponse) => {
                        // Cache successful responses
                        if (networkResponse && networkResponse.ok && networkResponse.type === 'basic') {
                            const responseToCache = networkResponse.clone();
                            caches.open(DYNAMIC_CACHE)
                                .then((cache) => cache.put(request, responseToCache));
                        }
                        return networkResponse;
                    })
                    .catch(() => {
                        // Offline fallback for navigation requests
                        if (request.mode === 'navigate') {
                            return caches.match('/index.html');
                        }
                        // Return simple offline response for other requests
                        return new Response('Çevrimdışı', {
                            status: 503,
                            statusText: 'Service Unavailable',
                            headers: { 'Content-Type': 'text/plain' }
                        });
                    });
            })
    );
});

// Handle push notifications
self.addEventListener('push', (event) => {
    console.log('[SW] Push received');

    let data = { title: 'SbYazılım Plasiyer', body: 'Yeni bildirim' };

    try {
        if (event.data) {
            data = event.data.json();
        }
    } catch (e) {
        console.error('[SW] Failed to parse push data:', e);
    }

    const options = {
        body: data.body || 'Yeni bildirim',
        icon: '/icons/icon-192x192.png',
        badge: '/icons/icon-72x72.png',
        vibrate: [100, 50, 100],
        data: data.data || {},
        actions: data.actions || [],
        tag: data.tag || 'plasiyer-notification',
        renotify: true,
        requireInteraction: data.requireInteraction || false,
    };

    event.waitUntil(
        self.registration.showNotification(data.title || 'SbYazılım Plasiyer', options)
    );
});

// Handle notification clicks
self.addEventListener('notificationclick', (event) => {
    console.log('[SW] Notification clicked');
    event.notification.close();

    const urlToOpen = event.notification.data?.url || '/';
    const action = event.action;

    // Handle action buttons
    if (action === 'view') {
        // Open specific page
    } else if (action === 'dismiss') {
        return; // Just close
    }

    event.waitUntil(
        self.clients.matchAll({ type: 'window', includeUncontrolled: true })
            .then((clientList) => {
                // Check if app is already open
                for (const client of clientList) {
                    if (client.url.includes(self.location.origin) && 'focus' in client) {
                        return client.focus().then(() => {
                            if ('navigate' in client) {
                                return client.navigate(urlToOpen);
                            }
                        });
                    }
                }
                // Open new window
                return self.clients.openWindow(urlToOpen);
            })
    );
});

// Handle notification close
self.addEventListener('notificationclose', (event) => {
    console.log('[SW] Notification closed');
});

// Background sync for offline actions
self.addEventListener('sync', (event) => {
    console.log('[SW] Background sync:', event.tag);

    if (event.tag === 'sync-orders') {
        event.waitUntil(syncOfflineOrders());
    }
    if (event.tag === 'sync-payments') {
        event.waitUntil(syncOfflinePayments());
    }
    if (event.tag === 'sync-all') {
        event.waitUntil(
            Promise.all([
                syncOfflineOrders(),
                syncOfflinePayments()
            ])
        );
    }
});

// Periodic background sync (for newer browsers)
self.addEventListener('periodicsync', (event) => {
    console.log('[SW] Periodic sync:', event.tag);

    if (event.tag === 'sync-data') {
        event.waitUntil(
            Promise.all([
                syncOfflineOrders(),
                syncOfflinePayments()
            ])
        );
    }
});

// Sync offline orders
async function syncOfflineOrders() {
    try {
        const cache = await caches.open(OFFLINE_QUEUE);
        const requests = await cache.keys();

        for (const request of requests) {
            if (request.url.includes('/orders')) {
                try {
                    const cachedData = await cache.match(request);
                    const data = await cachedData.json();

                    const response = await fetch('/api/orders', {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/json',
                            'X-Offline-Sync': 'true'
                        },
                        body: JSON.stringify(data)
                    });

                    if (response.ok) {
                        await cache.delete(request);
                        console.log('[SW] Synced offline order');

                        // Notify user
                        self.registration.showNotification('Sipariş Senkronize Edildi', {
                            body: 'Çevrimdışı oluşturulan sipariş başarıyla gönderildi.',
                            icon: '/icons/icon-192x192.png',
                            tag: 'sync-success'
                        });
                    }
                } catch (err) {
                    console.error('[SW] Failed to sync order:', err);
                }
            }
        }
    } catch (err) {
        console.error('[SW] Error in syncOfflineOrders:', err);
    }
}

// Sync offline payments
async function syncOfflinePayments() {
    try {
        const cache = await caches.open(OFFLINE_QUEUE);
        const requests = await cache.keys();

        for (const request of requests) {
            if (request.url.includes('/payments')) {
                try {
                    const cachedData = await cache.match(request);
                    const data = await cachedData.json();

                    const response = await fetch('/api/payments', {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/json',
                            'X-Offline-Sync': 'true'
                        },
                        body: JSON.stringify(data)
                    });

                    if (response.ok) {
                        await cache.delete(request);
                        console.log('[SW] Synced offline payment');

                        // Notify user
                        self.registration.showNotification('Tahsilat Senkronize Edildi', {
                            body: 'Çevrimdışı oluşturulan tahsilat başarıyla gönderildi.',
                            icon: '/icons/icon-192x192.png',
                            tag: 'sync-success'
                        });
                    }
                } catch (err) {
                    console.error('[SW] Failed to sync payment:', err);
                }
            }
        }
    } catch (err) {
        console.error('[SW] Error in syncOfflinePayments:', err);
    }
}

// Handle messages from main app
self.addEventListener('message', (event) => {
    console.log('[SW] Message received:', event.data);

    if (event.data && event.data.type === 'SKIP_WAITING') {
        self.skipWaiting();
    }

    if (event.data && event.data.type === 'GET_VERSION') {
        event.ports[0].postMessage({ version: CACHE_VERSION });
    }

    if (event.data && event.data.type === 'CLEAR_CACHE') {
        event.waitUntil(
            caches.keys().then((names) => {
                return Promise.all(
                    names.map((name) => caches.delete(name))
                );
            }).then(() => {
                event.ports[0].postMessage({ success: true });
            })
        );
    }
});

console.log('[SW] Service Worker loaded', CACHE_VERSION);
