import express from 'express';
import cors from 'cors';
import mysql from 'mysql2/promise';
import dotenv from 'dotenv';

dotenv.config();

const app = express();
const PORT = process.env.PORT || 3001;

// Middleware
app.use(cors());
app.use(express.json());

// Database connection pool - Optimized for serverless
const pool = mysql.createPool({
    host: process.env.DB_PLASIYER_HOST,
    user: process.env.DB_PLASIYER_USER,
    database: process.env.DB_PLASIYER_DATABASE,
    password: process.env.DB_PLASIYER_PASSWORD,
    port: process.env.DB_PLASIYER_PORT || 3306,
    waitForConnections: true,
    connectionLimit: 5,
    maxIdle: 2,
    idleTimeout: 30000,
    connectTimeout: 10000,
    enableKeepAlive: false,
});

// Auth helper - get user from header
function getUserFromHeader(req) {
    const authHeader = req.headers['x-auth-user'];
    if (!authHeader) return null;
    try {
        const decoded = Buffer.from(authHeader, 'base64').toString('utf8');
        return JSON.parse(decoded);
    } catch {
        return null;
    }
}

// Auth helper - get user info from DB
async function getUserInfo(kodu, sifre) {
    const [rows] = await pool.execute(
        'SELECT id, kodu, sirket, depo, sahibi, kasa FROM plasiyer WHERE kodu = ? AND sifre = ?',
        [kodu, sifre]
    );
    return rows.length > 0 ? rows[0] : null;
}

// Auth middleware
async function requireAuth(req, res, next) {
    const authData = getUserFromHeader(req);
    if (!authData || !authData.kodu || !authData.sifre) {
        return res.status(401).json({ error: 'Oturum bulunamadı' });
    }
    const user = await getUserInfo(authData.kodu, authData.sifre);
    if (!user) {
        return res.status(401).json({ error: 'Geçersiz kullanıcı' });
    }
    req.user = user;
    next();
}

// ==================== AUTH ROUTES ====================
app.post('/api/auth/login', async (req, res) => {
    const { kodu, sifre } = req.body;
    if (!kodu || !sifre) {
        return res.status(400).json({ error: 'Kullanıcı kodu ve şifre gereklidir.' });
    }
    try {
        const [rows] = await pool.execute(
            'SELECT * FROM plasiyer WHERE kodu = ? AND sifre = ?',
            [kodu, sifre]
        );
        if (rows.length > 0) {
            const user = rows[0];
            return res.json({
                success: true,
                user: {
                    id: user.id,
                    kodu: user.kodu,
                    adi: user.sahibi || user.adi || 'Kullanıcı',
                    sahibi: user.sahibi,
                    sirket: user.sirket,
                    depo: user.depo,
                    kasa: user.kasa
                }
            });
        } else {
            return res.status(401).json({ error: 'Hatalı kullanıcı kodu veya şifre.' });
        }
    } catch (error) {
        console.error('Login error:', error);
        return res.status(500).json({ error: 'Veritabanı bağlantı hatası: ' + error.message });
    }
});

app.get('/api/auth/check', requireAuth, (req, res) => {
    res.json({ authenticated: true, user: req.user });
});

app.post('/api/auth/logout', (req, res) => {
    res.json({ success: true });
});

// ==================== CUSTOMERS ROUTES ====================
app.get('/api/customers', requireAuth, async (req, res) => {
    try {
        const { limit = 50, search } = req.query;
        let query = 'SELECT id, carikodu, cariadi, bakiye, durum, sfuygulamasi, iskonto, ozelkod FROM carikartlar WHERE sirket = ? AND plasiyerid = ? AND kullanimda = ?';
        const params = [req.user.sirket, req.user.id, 'X'];

        if (search) {
            query += ' AND (cariadi LIKE ? OR carikodu LIKE ?)';
            params.push(`%${search}%`, `%${search}%`);
        }
        query += ' ORDER BY cariadi ASC LIMIT ?';
        params.push(parseInt(limit) || 50);

        const [rows] = await pool.execute(query, params);
        res.json(rows);
    } catch (error) {
        console.error('Customers error:', error);
        res.status(500).json({ error: 'Veritabanı hatası: ' + error.message });
    }
});

app.get('/api/customers/:id', requireAuth, async (req, res) => {
    try {
        const [rows] = await pool.execute(
            `SELECT id, carikodu, cariadi, telefon, fax, internet, ilgilikisi,
                    bakiye, durum, sfuygulamasi, iskonto, ozelkod, plasiyerid, sirket
             FROM carikartlar WHERE id = ? AND sirket = ? AND plasiyerid = ?`,
            [req.params.id, req.user.sirket, req.user.id]
        );

        if (rows.length === 0) {
            return res.status(404).json({ error: 'Müşteri bulunamadı' });
        }

        const customer = rows[0];

        // İletişim bilgilerini çek
        let iletisim = null;
        try {
            const [contacts] = await pool.execute(
                'SELECT id, yetkili, telefon, mail FROM iletisimbilgileri WHERE ckid = ? AND sirket = ? ORDER BY varsayilan DESC LIMIT 1',
                [customer.id, req.user.sirket]
            );
            if (contacts.length > 0) {
                iletisim = contacts[0];
            }
        } catch (e) {
            console.warn('İletişim bilgileri alınamadı:', e.message);
        }

        // Tüm iletişim bilgilerini çek
        let iletisimler = [];
        try {
            const [allContacts] = await pool.execute(
                'SELECT id, yetkili, telefon, mail, varsayilan FROM iletisimbilgileri WHERE ckid = ? AND sirket = ? ORDER BY varsayilan DESC',
                [customer.id, req.user.sirket]
            );
            iletisimler = allContacts;
        } catch (e) {
            console.warn('İletişim listesi alınamadı:', e.message);
        }

        // Adres bilgilerini çek
        let adresler = [];
        try {
            const [addresses] = await pool.execute(
                'SELECT id, adrestipi, adres1, adres2, ilcesi, ili, vergidairesi, vergino FROM adresler WHERE carikodu = ? AND sirket = ?',
                [customer.carikodu, req.user.sirket]
            );
            adresler = addresses;
        } catch (e) {
            console.warn('Adres bilgileri alınamadı:', e.message);
        }

        res.json({
            ...customer,
            iletisim,
            iletisimler,
            adresler
        });
    } catch (error) {
        console.error('Customer detail error:', error);
        res.status(500).json({ error: 'Veritabanı hatası: ' + error.message });
    }
});

app.get('/api/customers/:id/addresses', requireAuth, async (req, res) => {
    try {
        const [customers] = await pool.execute(
            "SELECT carikodu FROM carikartlar WHERE id = ? AND sirket = ? AND plasiyerid = ? AND kullanimda = 'X'",
            [req.params.id, req.user.sirket, req.user.id]
        );
        if (customers.length === 0) {
            return res.status(404).json({ error: 'Müşteri bulunamadı' });
        }
        const [addresses] = await pool.execute(
            'SELECT id, sirket, carikodu, adres1, adres2 FROM adresler WHERE carikodu = ? AND sirket = ? ORDER BY id ASC',
            [customers[0].carikodu, req.user.sirket]
        );
        res.json(addresses);
    } catch (error) {
        res.status(500).json({ error: 'Veritabanı hatası: ' + error.message });
    }
});

// Müşteri Siparişleri
app.get('/api/customers/:id/orders', requireAuth, async (req, res) => {
    try {
        const [customer] = await pool.execute(
            "SELECT carikodu FROM carikartlar WHERE id = ? AND sirket = ? AND plasiyerid = ? AND kullanimda = 'X'",
            [req.params.id, req.user.sirket, req.user.id]
        );
        if (customer.length === 0) {
            return res.status(404).json({ error: 'Müşteri bulunamadı' });
        }
        const [orders] = await pool.execute(
            `SELECT O.id, O.tarih, O.belgeno, O.tltutari, O.aciklama, O.katagori, O.adreskodu,
                    A.adres1 as teslimat_adresi
             FROM stokfb O
             LEFT JOIN adresler A ON O.adreskodu = A.id AND O.sirket = A.sirket
             WHERE O.carikodu = ? AND O.sirket = ? AND O.fistipkodu = '91'
             ORDER BY O.tarih DESC LIMIT 50`,
            [customer[0].carikodu, req.user.sirket]
        );
        res.json(orders);
    } catch (error) {
        res.status(500).json({ error: 'Veritabanı hatası: ' + error.message });
    }
});

app.get('/api/customers/:id/statement', requireAuth, async (req, res) => {
    try {
        const [customer] = await pool.execute(
            'SELECT carikodu, cariadi FROM carikartlar WHERE id = ? AND sirket = ? AND plasiyerid = ?',
            [req.params.id, req.user.sirket, req.user.id]
        );
        if (customer.length === 0) {
            return res.status(404).json({ error: 'Müşteri bulunamadı' });
        }
        const { carikodu, cariadi } = customer[0];

        const [transactionsRaw] = await pool.execute(
            "SELECT id, tarih, fistipi, fisadi, aciklama, fisno as belgeno, COALESCE(tborc, 0) as borc, COALESCE(talacak, 0) as alacak FROM carifisler WHERE carikodu = ? AND sirket = ? AND kgy = 'K' ORDER BY tarih ASC, id ASC LIMIT 100",
            [carikodu, req.user.sirket]
        );

        // Bakiye hesapla (kümülatif)
        let runningBalance = 0;
        const transactions = transactionsRaw.map(tx => {
            runningBalance += (parseFloat(tx.alacak) || 0) - (parseFloat(tx.borc) || 0);
            return {
                ...tx,
                borc: parseFloat(tx.borc) || 0,
                alacak: parseFloat(tx.alacak) || 0,
                bakiye: runningBalance
            };
        });

        const [totals] = await pool.execute(
            "SELECT COALESCE(SUM(tborc), 0) as borc, COALESCE(SUM(talacak), 0) as alacak FROM carifisler WHERE carikodu = ? AND sirket = ? AND kgy = 'K'",
            [carikodu, req.user.sirket]
        );

        res.json({
            success: true,
            customer: { id: req.params.id, carikodu, cariadi },
            transactions,
            totals: {
                borc: parseFloat(totals[0].borc) || 0,
                alacak: parseFloat(totals[0].alacak) || 0,
                bakiye: (parseFloat(totals[0].alacak) || 0) - (parseFloat(totals[0].borc) || 0)
            }
        });
    } catch (error) {
        res.status(500).json({ error: 'Veritabanı hatası: ' + error.message });
    }
});

// Müşteri Stok Hareketleri - PHP'deki stokdokumleri.php gibi
app.get('/api/customers/:id/stock-movements', requireAuth, async (req, res) => {
    try {
        const [customer] = await pool.execute(
            "SELECT carikodu, cariadi FROM carikartlar WHERE id = ? AND sirket = ? AND plasiyerid = ?",
            [req.params.id, req.user.sirket, req.user.id]
        );
        if (customer.length === 0) {
            return res.status(404).json({ error: 'Müşteri bulunamadı' });
        }
        const { carikodu, cariadi } = customer[0];

        const [movements] = await pool.execute(
            `SELECT id, tarih, fistipkodu, stokkodu, stokadi, miktar, birim1, tfiyat, 
                    COALESCE(isk1, 0) as isk1, COALESCE(isk2, 0) as isk2, 
                    COALESCE(isk3, 0) as isk3, COALESCE(isk4, 0) as isk4,
                    COALESCE(kdvlifiyat, 0) as toplam
             FROM stokfd 
             WHERE carikodu = ? AND sirket = ? 
             ORDER BY tarih DESC, id DESC 
             LIMIT 100`,
            [carikodu, req.user.sirket]
        );

        res.json({
            success: true,
            customer: { id: req.params.id, carikodu, cariadi },
            movements: movements.map(m => ({
                ...m,
                miktar: parseFloat(m.miktar) || 0,
                tfiyat: parseFloat(m.tfiyat) || 0,
                toplam: parseFloat(m.toplam) || 0
            }))
        });
    } catch (error) {
        res.status(500).json({ error: 'Veritabanı hatası: ' + error.message });
    }
});

// Müşteri Notları
app.get('/api/customers/:id/notes', requireAuth, async (req, res) => {
    try {
        const [customer] = await pool.execute(
            "SELECT carikodu FROM carikartlar WHERE id = ? AND sirket = ? AND plasiyerid = ?",
            [req.params.id, req.user.sirket, req.user.id]
        );
        if (customer.length === 0) {
            return res.status(404).json({ error: 'Müşteri bulunamadı' });
        }
        const { carikodu } = customer[0];

        console.log(`[DEBUG] Fetching notes for carikodu: ${carikodu}`);

        const [notes] = await pool.execute(
            'SELECT id, tarih, konu, memo FROM notlar WHERE kodu = ? AND sirket = ? AND plasiyerid = ? ORDER BY tarih DESC',
            [carikodu, req.user.sirket, req.user.id]
        );

        console.log(`[DEBUG] Found ${notes.length} notes`);
        res.json(notes);
    } catch (error) {
        res.status(500).json({ error: 'Veritabanı hatası: ' + error.message });
    }
});

// ==================== PRODUCTS ROUTES ====================
app.get('/api/products', requireAuth, async (req, res) => {
    try {
        const { limit = 100, grupkodu, search } = req.query;
        let query = "SELECT id, skodu, sadi, fiyat3, fark, birim1, carpan1, kdv, iskonto, tipk1 FROM stokkarti WHERE sirket = ? AND kullanimda = 'X'";
        const params = [req.user.sirket];

        if (grupkodu) {
            query += ' AND tipk1 = ?';
            params.push(grupkodu);
        }
        if (search) {
            query += ' AND (sadi LIKE ? OR skodu LIKE ?)';
            params.push(`%${search}%`, `%${search}%`);
        }
        query += ' ORDER BY sadi ASC LIMIT ?';
        params.push(parseInt(limit) || 100);

        const [rows] = await pool.execute(query, params);
        res.json(rows);
    } catch (error) {
        res.status(500).json({ error: 'Veritabanı hatası: ' + error.message });
    }
});

app.get('/api/products/groups', requireAuth, async (req, res) => {
    try {
        // Use tipa1 column for group name (exists in stokkarti table)
        const [rows] = await pool.execute(
            `SELECT tipk1 as grupkodu,
                    COALESCE(MAX(tipa1), tipk1) as grupadi,
                    COUNT(*) as adet
             FROM stokkarti
             WHERE sirket = ? AND kullanimda = 'X' AND tipk1 IS NOT NULL AND tipk1 != ''
             GROUP BY tipk1
             ORDER BY grupadi`,
            [req.user.sirket]
        );
        res.json(rows);
    } catch (error) {
        res.status(500).json({ error: 'Veritabanı hatası: ' + error.message });
    }
});

// Product detail endpoint
app.get('/api/products/:id', requireAuth, async (req, res) => {
    try {
        const [rows] = await pool.execute(
            "SELECT id, skodu, sadi, fiyat1, fiyat2, fiyat3, fark, birim1, birim2, carpan1, carpan2, kdv, iskonto, tipk1, tipk2 FROM stokkarti WHERE id = ? AND sirket = ? AND kullanimda = 'X'",
            [req.params.id, req.user.sirket]
        );
        if (rows.length === 0) {
            return res.status(404).json({ error: 'Ürün bulunamadı' });
        }
        res.json(rows[0]);
    } catch (error) {
        res.status(500).json({ error: 'Veritabanı hatası: ' + error.message });
    }
});

// ==================== ORDERS ROUTES ====================
app.get('/api/orders', requireAuth, async (req, res) => {
    try {
        const { limit = 50 } = req.query;
        const [rows] = await pool.execute(
            `SELECT O.id, O.tarih, O.belgeno, O.carikodu, O.cariadi, O.tltutari, O.aciklama, O.katagori, O.adreskodu,
                    A.adres1 as teslimat_adresi
             FROM stokfb O
             LEFT JOIN adresler A ON O.adreskodu = A.id AND O.sirket = A.sirket
             WHERE O.sirket = ? AND O.fistipkodu = ? AND O.plasiyerid = ?
             ORDER BY O.tarih DESC LIMIT ?`,
            [req.user.sirket, '91', req.user.id, parseInt(limit) || 50]
        );
        res.json(rows);
    } catch (error) {
        res.status(500).json({ error: 'Veritabanı hatası: ' + error.message });
    }
});

app.post('/api/orders', requireAuth, async (req, res) => {
    try {
        const { carikodu, cariadi, adresId, aciklama, tarih, belgeno } = req.body;
        if (!carikodu) {
            return res.status(400).json({ error: 'Müşteri kodu gerekli' });
        }

        // Müşteri kartından otomatik alanları çek
        const [musteriData] = await pool.execute(
            'SELECT sfuygulamasi, iskonto, ozelkod FROM carikartlar WHERE carikodu = ? AND sirket = ?',
            [carikodu, req.user.sirket]
        );

        const sfuygulamasi = musteriData[0]?.sfuygulamasi || ' ';
        const ciskonto = musteriData[0]?.iskonto || 0;
        const ozelkod = musteriData[0]?.ozelkod || '';

        // Tarih: Yerel saat kullan (PHP ile uyumlu - date("Y-m-d"))
        const now = new Date();
        const siparisTarihi = tarih || `${now.getFullYear()}-${String(now.getMonth() + 1).padStart(2, '0')}-${String(now.getDate()).padStart(2, '0')}`;

        // Saat: PHP'de sabit "01:00:00" kullanılıyor
        const saat = '01:00:00';

        const [result] = await pool.execute(
            `INSERT INTO stokfb (
                fistipkodu, fistipadi, carikodu, cariadi, adreskodu, adreskodu2, tarih, saat,
                vade, vadetarihi, depokodu, depoadi, aciklama, belgeno, ozelkod, dtip, dkur,
                bmatrah, nmatrah, iskonto, kdv, tltutari, dvztutari, sfuygulamasi, ciskonto,
                otv, gc, katagori, faturano, faturaid, ba, yazicikodu, sirket, plasiyerid
            ) VALUES (
                '91', 'Alinan Siparisler', ?, ?, ?, ?, ?, ?,
                '1', '2099-12-31', ?, ?, ?, ?, ?, ' ', 1,
                0, 0, 0, 0, 0, 0, ?, ?,
                0, '0', '7', '0', 0, 'A', 'A1', ?, ?
            )`,
            [
                carikodu, cariadi || '', adresId || 0, adresId || 0, siparisTarihi, saat,
                req.user.depo || '', req.user.sahibi || '', aciklama || '', belgeno || 'P', ozelkod,
                sfuygulamasi, ciskonto, req.user.sirket, req.user.id
            ]
        );

        res.status(201).json({ success: true, orderId: result.insertId });
    } catch (error) {
        console.error('Order creation error:', error);
        res.status(500).json({ error: 'Veritabanı hatası: ' + error.message });
    }
});

app.get('/api/orders/:id', requireAuth, async (req, res) => {
    try {
        const [orders] = await pool.execute(
            `SELECT O.id, O.tarih, O.saat, O.belgeno, O.carikodu, O.cariadi, O.adreskodu, O.tltutari, O.aciklama, O.katagori, 
                    O.bmatrah, O.nmatrah, O.iskonto, O.kdv,
                    C.telefon as musteri_telefon,
                    (SELECT telefon FROM iletisimbilgileri WHERE ckid = C.id LIMIT 1) as iletisim_telefon
             FROM stokfb O
             LEFT JOIN carikartlar C ON O.carikodu = C.carikodu AND O.sirket = C.sirket
             WHERE O.id = ? AND O.sirket = ? AND O.plasiyerid = ? AND O.fistipkodu = '91'`,
            [req.params.id, req.user.sirket, req.user.id]
        );
        if (orders.length === 0) {
            return res.status(404).json({ error: 'Sipariş bulunamadı' });
        }

        const order = orders[0];

        // Adres bilgisini çek
        let address = null;
        if (order.adreskodu) {
            const [addresses] = await pool.execute(
                'SELECT id, adres1, adres2 FROM adresler WHERE id = ? AND sirket = ?',
                [order.adreskodu, req.user.sirket]
            );
            if (addresses.length > 0) {
                address = addresses[0];
            }
        }

        const [items] = await pool.execute(
            'SELECT id, stokkodu, stokadi, miktar, birim1, tfiyat, netfiyat, kdvlifiyat, brutfiyat, kdv, isk1, isk2 FROM stokfd WHERE fisno = ? AND sirket = ? ORDER BY id ASC',
            [order.id, req.user.sirket]
        );
        // Frontend'in beklediği format: { header, items, address }
        res.json({ success: true, header: order, items: items, address: address });
    } catch (error) {
        res.status(500).json({ error: 'Veritabanı hatası: ' + error.message });
    }
});

app.post('/api/orders/:id/items', requireAuth, async (req, res) => {
    try {
        const orderId = req.params.id;
        const { stokkodu, stokadi, birim1, carpan1, miktar, tlfiyat, kdv, isk1, isk2 } = req.body;

        if (!stokkodu || !miktar || !tlfiyat) {
            return res.status(400).json({ error: 'Stok kodu, miktar ve fiyat gerekli' });
        }

        // Sipariş başlığını kontrol et (PHP stokekle.php ile uyumlu - tüm gerekli alanlar)
        const [orders] = await pool.execute(
            'SELECT id, tarih, saat, carikodu, depokodu, depoadi, fistipkodu, fistipadi, sirket, plasiyerid, ciskonto, faturano, gc, faturaid, katagori FROM stokfb WHERE id = ? AND sirket = ? AND plasiyerid = ?',
            [orderId, req.user.sirket, req.user.id]
        );

        if (orders.length === 0) {
            return res.status(404).json({ error: 'Sipariş bulunamadı' });
        }

        const order = orders[0];

        // Hesaplamalar
        const miktarVal = parseFloat(miktar) || 0;
        const tlfiyatVal = parseFloat(tlfiyat) || 0;
        const kdvVal = parseFloat(kdv) || 18;
        const isk1Val = parseFloat(isk1) || 0;
        const isk2Val = parseFloat(isk2) || parseFloat(order.ciskonto) || 0;

        const brutfiyat = miktarVal * tlfiyatVal;
        const isktutari1 = (brutfiyat / 100) * isk1Val;
        const aratoplam = brutfiyat - isktutari1;
        const isktutari2 = (aratoplam / 100) * isk2Val;
        const isktoplami = isktutari1 + isktutari2;
        const netfiyat = brutfiyat - isktoplami;
        const kdvtutari = (netfiyat / 100) * kdvVal;
        const kdvlifiyat = netfiyat + kdvtutari;

        // PHP stokekle.php ile uyumlu - 41 kolon
        const otvtutari = 0;
        const dvztutari = 0;

        const [insertResult] = await pool.execute(
            `INSERT INTO stokfd (
                fisno, tarih, saat, carikodu, kartno, stokkodu, stokadi,
                depokodu, depoadi, fistipkodu, fistipadi,
                miktar, birim1, carpan1, birim2, carpan2, dfiyat, tfiyat,
                isk1, isk2, isk3, isk4, isktutari,
                kdvtutari, otvtutari, netfiyat, kdvlifiyat, dvztutari, brutfiyat,
                serino, kdv, otv, scarpan1, scarpan2,
                faturano, gc, faturaid, shift,
                plasiyerid, sirket, katagori
            ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`,
            [
                orderId, order.tarih, order.saat || '01:00:00', order.carikodu, '', stokkodu, stokadi || '',
                order.depokodu || '', order.depoadi || '', order.fistipkodu || '91', order.fistipadi || 'SİPARİŞ',
                miktarVal, birim1 || 'AD', carpan1 || 1, ' ', 0, 0, tlfiyatVal,
                isk1Val, isk2Val, 0, 0, isktoplami,
                kdvtutari, otvtutari, netfiyat, kdvlifiyat, dvztutari, brutfiyat,
                ' ', kdvVal, 0, 1, 1,
                order.faturano || '0', order.gc || '0', order.faturaid || 0, ' ',
                req.user.id, req.user.sirket, order.katagori || '7'
            ]
        );

        // Sipariş toplamlarını güncelle (PHP stokekle.php ile uyumlu - otv ve dvztutari dahil)
        const [totals] = await pool.execute(
            'SELECT COALESCE(SUM(brutfiyat), 0) as sbrutfiyat, COALESCE(SUM(isktutari), 0) as sisktutari, COALESCE(SUM(netfiyat), 0) as snetfiyat, COALESCE(SUM(kdvtutari), 0) as skdvtutari, COALESCE(SUM(otvtutari), 0) as sotvtutari, COALESCE(SUM(kdvlifiyat), 0) as skdvlifiyat, COALESCE(SUM(dvztutari), 0) as sdvztutari FROM stokfd WHERE fisno = ?',
            [orderId]
        );

        const t = totals[0];
        await pool.execute(
            'UPDATE stokfb SET bmatrah = ?, iskonto = ?, nmatrah = ?, kdv = ?, otv = ?, tltutari = ?, dvztutari = ? WHERE id = ?',
            [t.sbrutfiyat, t.sisktutari, t.snetfiyat, t.skdvtutari, t.sotvtutari, t.skdvlifiyat, t.sdvztutari, orderId]
        );

        res.status(201).json({ success: true, itemId: insertResult.insertId });
    } catch (error) {
        console.error('Order items error:', error);
        res.status(500).json({ error: 'Veritabanı hatası: ' + error.message });
    }
});

// Sipariş Kalem Silme - PHP'nin stoksil.php'si gibi
app.delete('/api/orders/:orderId/items/:itemId', requireAuth, async (req, res) => {
    try {
        const { orderId, itemId } = req.params;

        // Önce kalemi sil
        await pool.execute(
            'DELETE FROM stokfd WHERE id = ? AND fisno = ? AND sirket = ?',
            [itemId, orderId, req.user.sirket]
        );

        // Sipariş toplamlarını yeniden hesapla (PHP'deki gibi)
        const [totals] = await pool.execute(
            'SELECT COALESCE(SUM(brutfiyat), 0) as sbrutfiyat, COALESCE(SUM(isktutari), 0) as sisktutari, COALESCE(SUM(netfiyat), 0) as snetfiyat, COALESCE(SUM(kdvtutari), 0) as skdvtutari, COALESCE(SUM(kdvlifiyat), 0) as skdvlifiyat, COALESCE(SUM(dvztutari), 0) as sdvztutari FROM stokfd WHERE fisno = ?',
            [orderId]
        );

        const t = totals[0];
        await pool.execute(
            'UPDATE stokfb SET bmatrah = ?, iskonto = ?, nmatrah = ?, kdv = ?, tltutari = ?, dvztutari = ? WHERE id = ?',
            [t.sbrutfiyat, t.sisktutari, t.snetfiyat, t.skdvtutari, t.skdvlifiyat, t.sdvztutari, orderId]
        );

        res.json({ success: true });
    } catch (error) {
        res.status(500).json({ error: 'Veritabanı hatası: ' + error.message });
    }
});

// ==================== PAYMENTS ROUTES ====================
app.get('/api/payments', requireAuth, async (req, res) => {
    try {
        const { startDate, endDate, limit = 50 } = req.query;
        let query = `SELECT F.id, F.tarih, F.carikodu, T.cariadi as cariunvani, F.tborc as tutar, F.aciklama, F.fistipi as tip
            FROM carifisler F LEFT JOIN carikartlar T ON F.carikodu = T.carikodu AND F.sirket = T.sirket
            WHERE F.sirket = ? AND F.plasiyerid = ? AND F.fistipi = '09'`;
        const params = [req.user.sirket, req.user.id];

        if (startDate) { query += ' AND F.tarih >= ?'; params.push(startDate); }
        if (endDate) { query += ' AND F.tarih <= ?'; params.push(endDate); }

        query += ' ORDER BY F.tarih DESC LIMIT ?';
        params.push(parseInt(limit) || 50);

        const [rows] = await pool.execute(query, params);
        res.json(rows);
    } catch (error) {
        res.status(500).json({ error: 'Veritabanı hatası: ' + error.message });
    }
});

app.post('/api/payments', requireAuth, async (req, res) => {
    try {
        const { carikodu, tutar, tip, aciklama } = req.body;
        if (!carikodu || !tutar) {
            return res.status(400).json({ error: 'Müşteri kodu ve tutar gerekli' });
        }
        const tarih = new Date().toISOString().split('T')[0];
        const [result] = await pool.execute(
            "INSERT INTO carifisler (tarih, carikodu, fistipi, tborc, talacak, aciklama, sirket, plasiyerid, kgy) VALUES (?, ?, '09', ?, 0, ?, ?, ?, 'K')",
            [tarih, carikodu, parseFloat(tutar), aciklama || 'Tahsilat', req.user.sirket, req.user.id]
        );
        res.status(201).json({ success: true, id: result.insertId });
    } catch (error) {
        res.status(500).json({ error: 'Veritabanı hatası: ' + error.message });
    }
});

// ==================== NOTES ROUTES ====================
// /api/notes/all - Tüm notları al (frontend bunu kullanıyor)
app.get('/api/notes/all', requireAuth, async (req, res) => {
    try {
        const [rows] = await pool.execute(
            'SELECT id, tarih, kodu, konu, memo FROM notlar WHERE sirket = ? AND plasiyerid = ? ORDER BY tarih DESC LIMIT 100',
            [req.user.sirket, req.user.id]
        );
        // Varsayılan durum ve oncelik değerlerini uygulama tarafında ata
        const notesWithDefaults = rows.map(note => ({
            ...note,
            durum: 'beklemede',
            oncelik: 'orta'
        }));
        res.json(notesWithDefaults);
    } catch (error) {
        res.status(500).json({ error: 'Veritabanı hatası: ' + error.message });
    }
});

app.get('/api/notes', requireAuth, async (req, res) => {
    try {
        const { carikodu } = req.query;
        let query = 'SELECT id, tarih, kodu, konu, memo FROM notlar WHERE sirket = ? AND plasiyerid = ?';
        const params = [req.user.sirket, req.user.id];
        if (carikodu) { query += ' AND kodu = ?'; params.push(carikodu); }
        query += ' ORDER BY tarih DESC LIMIT 50';
        const [rows] = await pool.execute(query, params);
        res.json(rows);
    } catch (error) {
        res.status(500).json({ error: 'Veritabanı hatası: ' + error.message });
    }
});

app.post('/api/notes', requireAuth, async (req, res) => {
    try {
        const { konu, memo, carikodu } = req.body;
        if (!konu) {
            return res.status(400).json({ error: 'Konu gerekli' });
        }
        const tarih = new Date().toISOString().split('T')[0];
        const [result] = await pool.execute(
            'INSERT INTO notlar (tarih, kodu, konu, memo, sirket, plasiyerid) VALUES (?, ?, ?, ?, ?, ?)',
            [tarih, carikodu || '', konu, memo || '', req.user.sirket, req.user.id]
        );
        res.status(201).json({ success: true, id: result.insertId });
    } catch (error) {
        res.status(500).json({ error: 'Veritabanı hatası: ' + error.message });
    }
});

app.post('/api/notes/:id/complete', requireAuth, async (req, res) => {
    try {
        await pool.execute(
            'UPDATE notlar SET durum = ? WHERE id = ? AND sirket = ? AND plasiyerid = ?',
            ['tamamlandi', req.params.id, req.user.sirket, req.user.id]
        );
        res.json({ success: true });
    } catch (error) {
        res.status(500).json({ error: 'Veritabanı hatası: ' + error.message });
    }
});

app.delete('/api/notes/:id', requireAuth, async (req, res) => {
    try {
        await pool.execute(
            'DELETE FROM notlar WHERE id = ? AND sirket = ? AND plasiyerid = ?',
            [req.params.id, req.user.sirket, req.user.id]
        );
        res.json({ success: true });
    } catch (error) {
        res.status(500).json({ error: 'Veritabanı hatası: ' + error.message });
    }
});

// ==================== DASHBOARD ROUTES ====================
app.get('/api/reports/dashboard', requireAuth, async (req, res) => {
    try {
        const today = new Date().toISOString().split('T')[0];
        const firstDayOfMonth = today.substring(0, 8) + '01';

        const [[todayOrders], [monthlyOrders], [todayPayments], [monthlyPayments], [customerStats], [recentOrders]] = await Promise.all([
            pool.execute("SELECT COUNT(*) as count, COALESCE(SUM(tltutari), 0) as total FROM stokfb WHERE sirket = ? AND plasiyerid = ? AND fistipkodu = '91' AND tarih = ?", [req.user.sirket, req.user.id, today]),
            pool.execute("SELECT COUNT(*) as count, COALESCE(SUM(tltutari), 0) as total FROM stokfb WHERE sirket = ? AND plasiyerid = ? AND fistipkodu = '91' AND tarih >= ?", [req.user.sirket, req.user.id, firstDayOfMonth]),
            pool.execute("SELECT COALESCE(SUM(talacak), 0) as total FROM carifisler WHERE sirket = ? AND plasiyerid = ? AND fistipi = '09' AND tarih = ?", [req.user.sirket, req.user.id, today]),
            pool.execute("SELECT COALESCE(SUM(talacak), 0) as total FROM carifisler WHERE sirket = ? AND plasiyerid = ? AND fistipi = '09' AND tarih >= ?", [req.user.sirket, req.user.id, firstDayOfMonth]),
            pool.execute("SELECT COUNT(*) as totalCustomers, COALESCE(SUM(CASE WHEN bakiye > 0 THEN bakiye ELSE 0 END), 0) as totalReceivable FROM carikartlar WHERE sirket = ? AND plasiyerid = ? AND kullanimda = 'X'", [req.user.sirket, req.user.id]),
            pool.execute("SELECT id, tarih, belgeno, carikodu, cariadi, tltutari FROM stokfb WHERE sirket = ? AND plasiyerid = ? AND fistipkodu = '91' ORDER BY tarih DESC, id DESC LIMIT 5", [req.user.sirket, req.user.id])
        ]);

        res.json({
            orders: {
                today: { count: todayOrders[0]?.count || 0, total: parseFloat(todayOrders[0]?.total) || 0 },
                monthly: { count: monthlyOrders[0]?.count || 0, total: parseFloat(monthlyOrders[0]?.total) || 0 }
            },
            payments: {
                today: { total: parseFloat(todayPayments[0]?.total) || 0 },
                monthly: { total: parseFloat(monthlyPayments[0]?.total) || 0 }
            },
            customers: {
                totalCustomers: customerStats[0]?.totalCustomers || 0,
                totalReceivable: parseFloat(customerStats[0]?.totalReceivable) || 0
            },
            recentOrders: recentOrders || []
        });
    } catch (error) {
        console.error('Dashboard error:', error);
        res.status(500).json({ error: 'Veritabanı hatası: ' + error.message });
    }
});

// Detailed Reports endpoint
app.get('/api/reports/detailed', requireAuth, async (req, res) => {
    try {
        const today = new Date().toISOString().split('T')[0];
        const firstDayOfMonth = today.substring(0, 8) + '01';
        const weekAgo = new Date(Date.now() - 7 * 24 * 60 * 60 * 1000).toISOString().split('T')[0];

        const [
            [todayOrders],
            [weeklyOrders],
            [monthlyOrders],
            [todayPayments],
            [weeklyPayments],
            [monthlyPayments],
            [customerStats],
            [topProducts],
            [topCustomers],
            [dailyOrders]
        ] = await Promise.all([
            // Orders
            pool.execute("SELECT COUNT(*) as count, COALESCE(SUM(tltutari), 0) as total FROM stokfb WHERE sirket = ? AND plasiyerid = ? AND fistipkodu = '91' AND tarih = ?", [req.user.sirket, req.user.id, today]),
            pool.execute("SELECT COUNT(*) as count, COALESCE(SUM(tltutari), 0) as total FROM stokfb WHERE sirket = ? AND plasiyerid = ? AND fistipkodu = '91' AND tarih >= ?", [req.user.sirket, req.user.id, weekAgo]),
            pool.execute("SELECT COUNT(*) as count, COALESCE(SUM(tltutari), 0) as total FROM stokfb WHERE sirket = ? AND plasiyerid = ? AND fistipkodu = '91' AND tarih >= ?", [req.user.sirket, req.user.id, firstDayOfMonth]),
            // Payments
            pool.execute("SELECT COALESCE(SUM(tborc), 0) as total FROM carifisler WHERE sirket = ? AND plasiyerid = ? AND fistipi = '09' AND tarih = ?", [req.user.sirket, req.user.id, today]),
            pool.execute("SELECT COALESCE(SUM(tborc), 0) as total FROM carifisler WHERE sirket = ? AND plasiyerid = ? AND fistipi = '09' AND tarih >= ?", [req.user.sirket, req.user.id, weekAgo]),
            pool.execute("SELECT COALESCE(SUM(tborc), 0) as total FROM carifisler WHERE sirket = ? AND plasiyerid = ? AND fistipi = '09' AND tarih >= ?", [req.user.sirket, req.user.id, firstDayOfMonth]),
            // Customers
            pool.execute("SELECT COUNT(*) as total, COALESCE(SUM(bakiye), 0) as totalReceivable FROM carikartlar WHERE sirket = ? AND plasiyerid = ? AND kullanimda = 'X'", [req.user.sirket, req.user.id]),
            // Top Products
            pool.execute("SELECT stokadi, SUM(miktar) as miktar, SUM(kdvlifiyat) as tutar FROM stokfd WHERE sirket = ? AND plasiyerid = ? AND tarih >= ? GROUP BY stokkodu, stokadi ORDER BY miktar DESC LIMIT 10", [req.user.sirket, req.user.id, firstDayOfMonth]),
            // Top Customers
            pool.execute("SELECT cariadi, COUNT(*) as siparis_sayisi, SUM(tltutari) as toplam_tutar FROM stokfb WHERE sirket = ? AND plasiyerid = ? AND fistipkodu = '91' AND tarih >= ? GROUP BY carikodu, cariadi ORDER BY toplam_tutar DESC LIMIT 10", [req.user.sirket, req.user.id, firstDayOfMonth]),
            // Daily orders for chart
            pool.execute("SELECT tarih, COUNT(*) as siparis, COALESCE(SUM(tltutari), 0) as tutar FROM stokfb WHERE sirket = ? AND plasiyerid = ? AND fistipkodu = '91' AND tarih >= ? GROUP BY tarih ORDER BY tarih ASC", [req.user.sirket, req.user.id, weekAgo])
        ]);

        res.json({
            orders: {
                today: { count: todayOrders[0]?.count || 0, total: parseFloat(todayOrders[0]?.total) || 0 },
                weekly: { count: weeklyOrders[0]?.count || 0, total: parseFloat(weeklyOrders[0]?.total) || 0 },
                monthly: { count: monthlyOrders[0]?.count || 0, total: parseFloat(monthlyOrders[0]?.total) || 0 }
            },
            payments: {
                today: { total: parseFloat(todayPayments[0]?.total) || 0 },
                weekly: { total: parseFloat(weeklyPayments[0]?.total) || 0 },
                monthly: { total: parseFloat(monthlyPayments[0]?.total) || 0 }
            },
            customers: {
                total: customerStats[0]?.total || 0,
                totalReceivable: parseFloat(customerStats[0]?.totalReceivable) || 0,
                activeThisMonth: topCustomers.length
            },
            topProducts: topProducts.map(p => ({
                stokadi: p.stokadi,
                miktar: parseFloat(p.miktar) || 0,
                tutar: parseFloat(p.tutar) || 0
            })),
            topCustomers: topCustomers.map(c => ({
                cariadi: c.cariadi,
                siparis_sayisi: c.siparis_sayisi || 0,
                toplam_tutar: parseFloat(c.toplam_tutar) || 0
            })),
            dailyOrders: dailyOrders.map(d => ({
                tarih: d.tarih,
                siparis: d.siparis || 0,
                tutar: parseFloat(d.tutar) || 0
            }))
        });
    } catch (error) {
        console.error('Detailed reports error:', error);
        res.status(500).json({ error: 'Veritabanı hatası: ' + error.message });
    }
});

// Static files - React build (production)
import path from 'path';
import { fileURLToPath } from 'url';

const __filename = fileURLToPath(import.meta.url);
const __dirname = path.dirname(__filename);

app.use(express.static(path.join(__dirname, 'dist')));

// SPA fallback - tüm diğer route'ları index.html'e yönlendir (Express 5 uyumlu)
app.get('/{*splat}', (req, res) => {
    res.sendFile(path.join(__dirname, 'dist', 'index.html'));
});

// Start server
app.listen(PORT, '0.0.0.0', () => {
    console.log(`🚀 Server running on http://0.0.0.0:${PORT}`);
    console.log(`📦 Database: ${process.env.DB_PLASIYER_HOST}:${process.env.DB_PLASIYER_PORT || 3306}`);
});

export default app;
